﻿using System;
using System.Drawing;
using System.Runtime.InteropServices;
using System.Windows.Forms;
using Microsoft.Win32;

namespace TaiVietKey
{
    public partial class frmMain : Form
    {
        private KeyboardHook hook;
        private bool isTypingEnabled = true;
        private const string appName = "TaiVietKey";

        private enum InputMode { None, TaiViet }
        private InputMode currentMode = InputMode.TaiViet;

        private const int HOTKEY_TAIVIET = 1;
        private const uint MOD_CONTROL = 0x0002;
        private const uint MOD_ALT = 0x0001;
        private const uint VK_X = 0x58;

        private NotifyIcon trayIcon;
        private ContextMenuStrip trayMenu;
        private HotkeyWindow hotkeyWindow;

        public string TextIcon;


        public frmMain()
        {
            InitializeComponent();
            hotkeyWindow = new HotkeyWindow();
            hotkeyWindow.OnToggleHotkeyPressed += ToggleInputMode;
            hotkeyWindow.RegisterHotkey();
            LoadStartupStatus();
            this.KeyPreview = true;

            // --- Hook bàn phím ---
            hook = new KeyboardHook();
            hook.OnKeyPressed += Hook_OnKeyPressed;
            hook.Start();

            // --- Icon khay hệ thống ---
            // --- Icon khay hệ thống ---
            trayMenu = new ContextMenuStrip();
            trayMenu.Items.Add("Mở TaiVietKey", null, (s, e) => ShowMainWindow());
            trayMenu.Items.Add("Thoát", null, (s, e) => ExitApp());

            // --- Tạo icon khay ---
            trayIcon = new NotifyIcon
            {
                Icon = CreateTrayIcon(currentMode == InputMode.TaiViet ? "T" : "A"),
                Text = currentMode == InputMode.TaiViet ? "TaiVietKey - Gõ chữ Thái" : "TaiVietKey - Gõ chữ thường",
                ContextMenuStrip = trayMenu,
                Visible = true
            };

            // 🟢 Xử lý click chuột
            trayIcon.MouseClick += (s, e) =>
            {
                if (e.Button == MouseButtons.Left)
                    ToggleInputMode(); // 🔹 Click đơn → đổi kiểu nhập
            };

            // 🟢 Xử lý double-click
            trayIcon.MouseDoubleClick += (s, e) =>
            {
                if (e.Button == MouseButtons.Left)
                    ShowMainWindow(); // 🔹 Double-click → mở cửa sổ chính
                    ToggleInputMode(); // 🔹 Double-click → đổi kiểu nhập
            };

        }

        // === TẠO ICON "T"/"A" DƯỚI KHAY ===
        private Icon CreateTrayIcon(string text)
        {
            using (Bitmap bmp = new Bitmap(32, 32))
            using (Graphics g = Graphics.FromImage(bmp))
            {
                g.SmoothingMode = System.Drawing.Drawing2D.SmoothingMode.AntiAlias;
                g.Clear(Color.Transparent);

                // --- Nền ---
                Color bgColor = text == "T" ? Color.FromArgb(0, 64, 0) : Color.Red;
                using (Brush bgBrush = new SolidBrush(bgColor))
                {
                    g.FillRectangle(bgBrush, 0, 0, 32, 32);
                }

                // --- Viền trắng ---
                using (Pen borderPen = new Pen(Color.White, 2))
                {
                    g.DrawRectangle(borderPen, 1, 1, 30, 30);
                }

                // --- Chữ trắng, căn giữa ---
                using (Font f = new Font("Arial", 18, FontStyle.Bold, GraphicsUnit.Pixel))
                using (Brush b = new SolidBrush(Color.White))
                {
                    var sf = new StringFormat { Alignment = StringAlignment.Center, LineAlignment = StringAlignment.Center };
                    g.DrawString(text, f, b, new RectangleF(0, 0, 32, 32), sf);
                }

                IntPtr hIcon = bmp.GetHicon();
                return Icon.FromHandle(hIcon);
            }
        }




        [DllImport("user32.dll")]
        private static extern short GetKeyState(int keyCode);
        private const int VK_SHIFT = 0x10;

        // === BẮT PHÍM TỪ HOOK ===
        private bool Hook_OnKeyPressed(Keys key)
        {
            bool isShiftPressed = (GetKeyState(VK_SHIFT) & 0x8000) != 0;
            if ((Control.ModifierKeys & Keys.Control) == Keys.Control) return false;
            if (!isTypingEnabled || currentMode == InputMode.None) return false;

            string output = null;

            // (CÁC BẢNG GÕ CỦA BẠN GIỮ NGUYÊN, chỉ ví dụ vài ký tự)
            switch (key)
            {
                //Hàng phím 123
                case Keys.Oemtilde:
                    output = isShiftPressed ? "ꪧ" : "ꪦ";
                    break;
                case Keys.D1:
                    output = isShiftPressed ? "ꫛ" : "ꫜ";
                    break;
                case Keys.D2:
                    output = isShiftPressed ? "ꫝ" : "ꪺ";
                    break;
                case Keys.D3:
                    output = isShiftPressed ? "꫟" : "ꪻ";
                    break;
                case Keys.D4:
                    output = isShiftPressed ? "ꪹ" : "ꪹ";
                    break;
                case Keys.D5:
                    output = isShiftPressed ? "꫚" : "꫞";
                    break;
                case Keys.D6:
                    output = isShiftPressed ? "ꪃ" : "ꪂ";
                    break;
                case Keys.D7:
                    output = isShiftPressed ? "ꪍ" : "ꪌ";
                    break;
                case Keys.D8:
                    output = isShiftPressed ? "ꪆ" : "ꪳ";
                    break;
                case Keys.D9:
                    output = isShiftPressed ? "ꪇ" : "ꪸ";
                    break;
                case Keys.D0:
                    output = isShiftPressed ? "ꪷ" : "ꪷ";
                    break;
                case Keys.OemMinus:
                    output = isShiftPressed ? "ꪰ" : "ꪰ";
                    break;
                case Keys.Oemplus:
                    output = isShiftPressed ? "ꪽ" : "ꪾ";
                    break;

                // Hàng phím QWE
                case Keys.Q:
                    output = isShiftPressed ? "ꪺ" : "ꪶ";
                    break;
                case Keys.W:
                    output = isShiftPressed ? "ꪻ" : "ꪼ";
                    break;
                case Keys.E:
                    output = isShiftPressed ? "꫃" : "ꪵ";
                    break;
                case Keys.R:
                    output = isShiftPressed ? "꫁" : "꫁";
                    break;
                case Keys.T:
                    output = isShiftPressed ? "ꪕ" : "ꪔ";
                    break;
                case Keys.Y:
                    output = isShiftPressed ? "ꪗ" : "ꪖ";
                    break;
                case Keys.U:
                    output = isShiftPressed ? "ꪳ" : "ꪴ";
                    break;
                case Keys.I:
                    output = isShiftPressed ? "ꪸ" : "ꪲ";
                    break;
                case Keys.O:
                    output = isShiftPressed ? "ꪯ" : "ꪮ";
                    break;
                case Keys.P:
                    output = isShiftPressed ? "ꪝ" : "ꪜ";
                    break;
                case Keys.OemOpenBrackets:
                    output = isShiftPressed ? "ꪡ" : "ꪠ";
                    break;
                case Keys.OemCloseBrackets:
                    output = isShiftPressed ? "ꪟ" : "ꪞ";
                    break;

                // Hàng phím ASD
                case Keys.A:
                    output = isShiftPressed ? "꫄" : "ꪱ";
                    break;
                case Keys.S:
                    output = isShiftPressed ? "ꪏ" : "ꪎ";
                    break;
                case Keys.D:
                    output = isShiftPressed ? "ꪓ" : "ꪒ";
                    break;
                case Keys.F:
                    output = isShiftPressed ? "꪿" : "꪿";
                    break;
                case Keys.G:
                    output = isShiftPressed ? "ꪈ" : "ꪉ";
                    break;
                case Keys.H:
                    output = isShiftPressed ? "ꪭ" : "ꪬ";
                    break;
                case Keys.J:
                    output = isShiftPressed ? "ꪤ" : "ꪥ";
                    break;
                case Keys.K:
                    output = isShiftPressed ? "ꪁ" : "ꪀ";
                    break;
                case Keys.L:
                    output = isShiftPressed ? "ꪨ" : "ꪩ";
                    break;

                // Hàng phím ZXC
                case Keys.Z:
                    output = isShiftPressed ? "ꪐ" : "ꪑ";
                    break;
                case Keys.X:
                    output = isShiftPressed ? "ꪅ" : "ꪄ";
                    break;
                case Keys.C:
                    output = isShiftPressed ? "ꪋ" : "ꪊ";
                    break;
                case Keys.V:
                    output = isShiftPressed ? "ꪪ" : "ꪫ";
                    break;
                case Keys.B:
                    output = isShiftPressed ? "ꪛ" : "ꪚ";
                    break;
                case Keys.N:
                    output = isShiftPressed ? "ꪘ" : "ꪙ";
                    break;
                case Keys.M:
                    output = isShiftPressed ? "ꪢ" : "ꪣ";
                    break;

            }

            if (output != null)
            {
                SendKeys.Send(output);
                return true;
            }
            return false;
        }

        // === NÚT BẬT/TẮT GÕ TAI VIỆT ===
        private void ToggleInputMode()
        {
            if (currentMode == InputMode.TaiViet)
            {
                currentMode = InputMode.None;
                isTypingEnabled = false;
                btnToggle.Text = "A";
                btnToggle.BackColor = Color.Red;
                trayIcon.Icon = CreateTrayIcon("A");
                trayIcon.Text = "TaiVietKey - Gõ chữ thường"; // ✅ cập nhật tooltip
            }
            else
            {
                currentMode = InputMode.TaiViet;
                isTypingEnabled = true;
                btnToggle.Text = "T";
                btnToggle.BackColor = Color.FromArgb(0, 64, 0);
                trayIcon.Icon = CreateTrayIcon("T");
                trayIcon.Text = "TaiVietKey - Gõ chữ Thái"; // ✅ cập nhật tooltip
            }
        }


        private void btnToggle_Click(object sender, EventArgs e)
        {
            ToggleInputMode();
        }

        private void btnHide_Click(object sender, EventArgs e) => this.WindowState = FormWindowState.Minimized;
        private void btnStop_Click(object sender, EventArgs e)
        {
            // ✅ Tắt hoàn toàn chương trình
            trayIcon.Visible = false;  // Ẩn icon khay
            hook.Stop();               // Dừng hook bàn phím
            UnregisterHotKey(this.Handle, HOTKEY_TAIVIET); // Hủy hotkey
            Application.Exit();        // Thoát chương trình
        }
        private void btnClose_Click(object sender, EventArgs e) => this.Close();

        // === KHI FORM ĐÓNG, CHỈ ẨN XUỐNG KHAY ===
        protected override void OnFormClosing(FormClosingEventArgs e)
        {
            if (e.CloseReason == CloseReason.UserClosing)
            {
                e.Cancel = true;
                this.WindowState = FormWindowState.Minimized;
                this.ShowInTaskbar = false;
                this.Visible = true; // ✅ Giữ handle để hotkeyWindow vẫn hoạt động
                return;
            }

            // ✅ Chỉ khi thật sự thoát mới hủy hotkey
            trayIcon.Visible = false;
            hook.Stop();
            hotkeyWindow.Dispose(); // di chuyển lên trước base.OnFormClosing
            UnregisterHotKey(this.Handle, HOTKEY_TAIVIET);
            base.OnFormClosing(e);
        }




        // === HIỂN THỊ LẠI FORM ===
        private void ShowMainWindow()
        {
            this.Show();
            this.WindowState = FormWindowState.Normal;
            this.ShowInTaskbar = true;
        }

        private void ExitApp()
        {
            trayIcon.Visible = false;
            hook.Stop();
            Application.Exit();
        }

        // === PHÍM TẮT CTRL+ALT+X ===
        private void frmMain_Load(object sender, EventArgs e)
        {
            // Đăng ký phím tắt Ctrl + Alt + X
            RegisterHotKey(this.Handle, HOTKEY_TAIVIET, MOD_CONTROL | MOD_ALT, VK_X);
        }

        protected override void WndProc(ref Message m)
        {
            const int WM_HOTKEY = 0x0312;

            if (m.Msg == WM_HOTKEY)
            {
                if (m.WParam.ToInt32() == HOTKEY_TAIVIET)
                {
                    ToggleInputMode(); // 🔹 Gọi hàm đổi kiểu gõ
                }
            }

            base.WndProc(ref m);
        }


        private void frmMain_KeyDown(object sender, KeyEventArgs e)
        {
            if (e.Control && e.Alt && e.KeyCode == Keys.X)
            {
                ToggleInputMode();
                e.Handled = true;
            }
        }

        // === KHỞI ĐỘNG CÙNG WINDOWS ===
        private void chkStartup_CheckedChanged(object sender, EventArgs e)
        {
            if (chkStartup.Checked) AddToStartup();
            else RemoveFromStartup();
        }

        private void LoadStartupStatus()
        {
            using (RegistryKey key = Registry.CurrentUser.OpenSubKey(@"Software\\Microsoft\\Windows\\CurrentVersion\\Run", false))
                chkStartup.Checked = key?.GetValue(appName) != null;
        }

        private void AddToStartup()
        {
            string exePath = Application.ExecutablePath;
            using (RegistryKey key = Registry.CurrentUser.OpenSubKey(@"Software\\Microsoft\\Windows\\CurrentVersion\\Run", true))
                key.SetValue(appName, exePath);
        }

        private void RemoveFromStartup()
        {
            using (RegistryKey key = Registry.CurrentUser.OpenSubKey(@"Software\\Microsoft\\Windows\\CurrentVersion\\Run", true))
                key?.DeleteValue(appName, false);
        }

        // === KHAI BÁO API ===
        [DllImport("user32.dll")] public static extern bool RegisterHotKey(IntPtr hWnd, int id, uint fsModifiers, uint vk);
        [DllImport("user32.dll")] public static extern bool UnregisterHotKey(IntPtr hWnd, int id);

        private void btnKeyMap_Click(object sender, EventArgs e)
        {
            frmKM_TaiViet keyMapForm = new frmKM_TaiViet();
            keyMapForm.ShowDialog();
        }
    }
}
